from fastapi import APIRouter, Depends, HTTPException, BackgroundTasks, Path
from fastapi.responses import JSONResponse
from typing import List, Optional

from app.db import database
from ...models.platform.workforcecategorymodel import (
    WorkforceCategory, WorkforceCategoryBase, WorkforceCategoryResponseList, WorkforceCategoryUpdate
)
from ...dependencies.auth import get_current_userdetails
from ...libraries.object import str_to_objectid

# Import service functions
from ...services.platform.workforcecategory import (
    create_workforcecategory_service,
    get_workforcecategories_service,
    read_workforcecategory_service,
    update_workforcecategory_service,
    delete_workforcecategory_service,
)

router = APIRouter()

@router.post("/", response_model=WorkforceCategory)
def create_workforcecategory_endpoint(
    workforcecategory: WorkforceCategoryBase,
    background_tasks: BackgroundTasks,
    db: database.MongoDB = Depends(database.get_mongo_db)
):
    """
    Creates a new workforce category.
    """
    result = create_workforcecategory_service(workforcecategory, background_tasks, db)
    return JSONResponse(content={"message": "Workforce category created successfully", "workforcecategory": result["workforcecategory_data"]})

@router.get("/list/", response_model=WorkforceCategoryResponseList)
def get_workforcecategories(
    skip: int = 0,
    limit: int = 10,
    q: Optional[str] = None,
    status: Optional[str] = None,
    sort_by: Optional[str] = None,         
    sort_order: Optional[str] = "asc",     
    db: database.MongoDB = Depends(database.get_mongo_db),
    current_user: dict = Depends(get_current_userdetails)
):
    """
    Returns a list of workforce categories.
    """
    return get_workforcecategories_service(skip, limit, q, status, sort_by, sort_order, db, current_user)

@router.get("/{workforcecategory_id}", response_model=WorkforceCategory)
def read_workforcecategory(
    workforcecategory_id: str,
    db: database.MongoDB = Depends(database.get_mongo_db)
):
    """
    Returns workforce category details by ID.
    """
    workforcecategory = read_workforcecategory_service(workforcecategory_id, db)
    if workforcecategory is None:
        raise HTTPException(status_code=404, detail="Workforce category not found")
    return workforcecategory

@router.put("/{workforcecategory_id}", response_model=WorkforceCategory)
def update_workforcecategory(
    workforcecategory_id: str,
    workforcecategory_data: WorkforceCategoryUpdate,
    db: database.MongoDB = Depends(database.get_mongo_db)
):
    """
    Updates workforce category information.
    """
    updated_workforcecategory = update_workforcecategory_service(workforcecategory_id, workforcecategory_data, db)
    if updated_workforcecategory is None:
        raise HTTPException(status_code=404, detail="Workforce category not found after update")
    return updated_workforcecategory

@router.delete("/{workforcecategory_id}", response_model=WorkforceCategory)
def delete_workforcecategory(
    workforcecategory_id: str = Path(..., description="The ID of the workforce category"),
    db: database.MongoDB = Depends(database.get_mongo_db)
):
    """
    Deletes a workforce category by ID.
    """
    workforcecategory = delete_workforcecategory_service(workforcecategory_id, db)
    if workforcecategory is None:
        raise HTTPException(status_code=404, detail="Workforce category not found")
    return workforcecategory